
#ifndef _MAPGEN_H
#define _MAPGEN_H

#include <windows.h>
#include <iostream.h>
#include <malloc.h>
#include <memory.h>
#include <string.h>
#include <math.h>


#ifndef _USEFUL_MACROS
#define _USEFUL_MACROS
const double PI = 3.14159265358979323846264338327950288419716939937510f;
const double PI2 = PI * 2;
const double RADIAN = PI / 180; 

#define SIN360(theta) (float)sin((double)theta * RADIAN) 
#define COS360(theta) (float)cos((double)theta * RADIAN) 
#define TAN360(theta) (float)tan((double)theta * RADIAN)
#define SWAP(a,b) ((a)^=(b)^=(a)^=(b))
#define RGB16(r,g,b) (USHORT)((b%32) + ((g%64) << 6) + ((r%32) << 11))
#define RGB32(a,r,g,b) (UINT)((b) + ((g) << 8) + ((r) << 16) + ((a) << 24))
#endif


typedef struct tagTRIANGLE
{
	float x1, y1, z1;
	float x2, y2, z2;
	float x3, y3, z3;

	int scr_x1, scr_y1;
	int scr_x2, scr_y2;
	int scr_x3, scr_y3;

	DWORD color;
}TRIANGLE;


class MapGen
{
public:

	MapGen();
   ~MapGen();

	void MakeHeightMap(int iterations, float seed, int rand_level);
	void TesselateMap(void);
	void ProjectMap(int scr_width, int scr_height, int flags);
	void RotateMap(float rx, float ry, float rz);
	void DistortMap(void);
	void ScaleMap(float mag);
	float GetValue(int x, int y);

	float *height_map;
	long  dimension;
	int   iterations;
	
	float proj_dist;

	long num_triangles;
	TRIANGLE *tri_list;

private:

	void  DiamondStep(int x, int y, int halfstep, int randomness);
	void  SquareStep(int x, int y, int halfstep, int randomness);
	void  SetValue(int x, int y, float value);
};


MapGen::MapGen()
{
	height_map = NULL;
	dimension = 0;
	iterations = 0;
	proj_dist = 0;
	num_triangles = 0;
	tri_list = NULL;
}


void MapGen::MakeHeightMap(int iterations, float seed, int rand_level)
{
	if(height_map != NULL) delete[] height_map;
	dimension = 0;

	proj_dist = ((float)rand_level * 20);

	int InitialStep = 32;
	int step, halfstep, randomness = rand_level;//InitialStep;
	int x, y;
	
	if(iterations > 11) iterations = 11;
	MapGen::iterations = iterations;
	dimension  = (long)(pow(2,iterations) + 1); //size = (2^i +1)^2
	height_map = new float[dimension * dimension];
	
	for(y=0; y<dimension; y++)
	{
		for(x=0; x<dimension; x++)
		{
			SetValue(x, y, 0);
		}
	}

	SetValue(0, 0, seed);
	SetValue(dimension-1, 0, seed);
	SetValue(0, dimension-1, seed);
	SetValue(dimension-1, dimension-1, seed);

	for(step=InitialStep; step>1; step /= 2)
	{
		halfstep = step/2;
	
		for(y=0; y<dimension; y+=step)
		{
			for(x=0; x<dimension; x+=step)
			{
				DiamondStep(x + halfstep, y + halfstep, halfstep, randomness);
			}
		}

		for(y=0; y<dimension; y+=step)
		{
			for(x=0; x<dimension; x+=step)
			{
				SquareStep(x + halfstep, y, halfstep, randomness);
				SquareStep(x, y + halfstep, halfstep, randomness);
			}
		}
		randomness /= 2;
	}
}



void MapGen::TesselateMap(void)
{
	if(height_map == NULL) return;
	if(tri_list != NULL) delete[] tri_list;

	int x, y;
	DWORD index = 0;

	num_triangles = (DWORD)pow(dimension-1,2) * 2;
	tri_list = new TRIANGLE[num_triangles];

	for(y=0; y<dimension-1; y++)
	{
		for(x=0; x<dimension-1; x++)
		{
			tri_list[index].x1 = (float)x;
			tri_list[index].y1 = (float)y;
			tri_list[index].z1 = GetValue(x, y);

			tri_list[index].x2 = (float)(x + 1);
			tri_list[index].y2 = (float)y;
			tri_list[index].z2 = GetValue(x + 1, y);

			tri_list[index].x3 = (float)x;
			tri_list[index].y3 = (float)(y + 1);
			tri_list[index].z3 = GetValue(x, y + 1);
			
			tri_list[index].color = (DWORD)(tri_list[index].z1 + tri_list[index].z2 + tri_list[index].z3) / 3;

			index++;

			tri_list[index].x1 = (float)x;
			tri_list[index].y1 = (float)(y + 1);
			tri_list[index].z1 = GetValue(x, y + 1);

			tri_list[index].x2 = (float)(x + 1);
			tri_list[index].y2 = (float)(y + 1);
			tri_list[index].z2 = GetValue(x + 1, y + 1);

			tri_list[index].x3 = (float)(x + 1);
			tri_list[index].y3 = (float)y;
			tri_list[index].z3 = GetValue(x + 1, y);

			tri_list[index].color = (DWORD)(tri_list[index].z1 + tri_list[index].z2 + tri_list[index].z3) / 3;

			index++;
		}
	}

}


void MapGen::ProjectMap(int scr_width, int scr_height, int flags)
{
	if(tri_list == NULL) return;

	float cx, cy;
	cx = (float)dimension / 2;
	cy = (float)dimension / 2;

	float dist = proj_dist;
	float hdist = dist / 2;

	float midx = (scr_width / 2) - cx;
	float midy = (scr_height / 2) - cy;
	
	long index;
	for(index=0; index<num_triangles; index++)
	{
		if(flags == 1)
		{
			tri_list[index].scr_x1 = (int)(((tri_list[index].x1 - hdist) * dist) / (tri_list[index].z1 + hdist)) + (scr_width / 2) + (int)(dist - cx); 
			tri_list[index].scr_y1 = (int)(((tri_list[index].y1 - hdist) * dist) / (tri_list[index].z1 + hdist)) + (scr_height / 2) + (int)(dist - cy); 
			
			tri_list[index].scr_x2 = (int)(((tri_list[index].x2 - hdist) * dist) / (tri_list[index].z2 + hdist)) + (scr_width / 2) + (int)(dist - cx); 
			tri_list[index].scr_y2 = (int)(((tri_list[index].y2 - hdist) * dist) / (tri_list[index].z2 + hdist)) + (scr_height / 2) + (int)(dist - cy); 

			tri_list[index].scr_x3 = (int)(((tri_list[index].x3 - hdist) * dist) / (tri_list[index].z3 + hdist)) + (scr_width / 2) + (int)(dist - cx); 
			tri_list[index].scr_y3 = (int)(((tri_list[index].y3 - hdist) * dist) / (tri_list[index].z3 + hdist)) + (scr_height / 2) + (int)(dist - cy); 
		}
		else
		{		
			tri_list[index].scr_x1 = (int)(tri_list[index].x1 + midx);
			tri_list[index].scr_y1 = (int)(tri_list[index].y1 + midy);
			
			tri_list[index].scr_x2 = (int)(tri_list[index].x2 + midx);
			tri_list[index].scr_y2 = (int)(tri_list[index].y2 + midy);

			tri_list[index].scr_x3 = (int)(tri_list[index].x3 + midx);
			tri_list[index].scr_y3 = (int)(tri_list[index].y3 + midy);
		}	
	}
}


void MapGen::RotateMap(float rx, float ry, float rz)
{
	long index;
	
	float cx, cy, cz;
	float tmp_x, tmp_y, tmp_z;
	cx = (float)dimension / 2;
	cy = (float)dimension / 2;
	cz = 0;

	const float cos_rx = COS360(rx), sin_rx = SIN360(rx), 
				cos_ry = COS360(ry), sin_ry = SIN360(ry), 
				cos_rz = COS360(rz), sin_rz = SIN360(rz);

	for(index=0; index<num_triangles; index++)
	{
		//make the point's origin and the center of rotation the same//
		tri_list[index].x1 -= cx;
		tri_list[index].y1 -= cy;
		tri_list[index].z1 -= cz;

		//X axis rotation, by 360-degree angle rx //
		tmp_y = tri_list[index].y1 * cos_rx - tri_list[index].z1 * sin_rx;
		tmp_z = tri_list[index].z1 * cos_rx + tri_list[index].y1 * sin_rx;
		tri_list[index].y1 = tmp_y;
		tri_list[index].z1 = tmp_z;

		//Y axis rotation, by 360-degree angle ry//
		tmp_z = tri_list[index].z1 * cos_ry - tri_list[index].x1 * sin_ry;
		tmp_x = tri_list[index].x1 * cos_ry + tri_list[index].z1 * sin_ry;
		tri_list[index].z1 = tmp_z;
		tri_list[index].x1 = tmp_x;

		//Z axis rotation, by 360-degree angle rz//
		tmp_x = tri_list[index].x1 * cos_rz - tri_list[index].y1 * sin_rz;
		tmp_y = tri_list[index].y1 * cos_rz + tri_list[index].x1 * sin_rz;
		tri_list[index].x1 = tmp_x;                                    
		tri_list[index].y1 = tmp_y;

		//Translate point origin back//
		tri_list[index].x1 += cx;
		tri_list[index].y1 += cy;
		tri_list[index].z1 += cz;		


		////////////////////////////////////////////////////////////////////////////////

		//make the point's origin and the center of rotation the same//
		tri_list[index].x2 -= cx;
		tri_list[index].y2 -= cy;
		tri_list[index].z2 -= cz;

		//X axis rotation, by 360-degree angle rx //
		tmp_y = tri_list[index].y2 * cos_rx - tri_list[index].z2 * sin_rx;
		tmp_z = tri_list[index].z2 * cos_rx + tri_list[index].y2 * sin_rx;
		tri_list[index].y2 = tmp_y;
		tri_list[index].z2 = tmp_z;

		//Y axis rotation, by 360-degree angle ry//
		tmp_z = tri_list[index].z2 * cos_ry - tri_list[index].x2 * sin_ry;
		tmp_x = tri_list[index].x2 * cos_ry + tri_list[index].z2 * sin_ry;
		tri_list[index].z2 = tmp_z;
		tri_list[index].x2 = tmp_x;

		//Z axis rotation, by 360-degree angle rz//
		tmp_x = tri_list[index].x2 * cos_rz - tri_list[index].y2 * sin_rz;
		tmp_y = tri_list[index].y2 * cos_rz + tri_list[index].x2 * sin_rz;
		tri_list[index].x2 = tmp_x;                                    
		tri_list[index].y2 = tmp_y;

		//Translate point origin back//
		tri_list[index].x2 += cx;
		tri_list[index].y2 += cy;
		tri_list[index].z2 += cz;		


		////////////////////////////////////////////////////////////////////////////////////////

		//make the point's origin and the center of rotation the same//
		tri_list[index].x3 -= cx;
		tri_list[index].y3 -= cy;
		tri_list[index].z3 -= cz;

		//X axis rotation, by 360-degree angle rx //
		tmp_y = tri_list[index].y3 * cos_rx - tri_list[index].z3 * sin_rx;
		tmp_z = tri_list[index].z3 * cos_rx + tri_list[index].y3 * sin_rx;
		tri_list[index].y3 = tmp_y;
		tri_list[index].z3 = tmp_z;

		//Y axis rotation, by 360-degree angle ry//
		tmp_z = tri_list[index].z3 * cos_ry - tri_list[index].x3 * sin_ry;
		tmp_x = tri_list[index].x3 * cos_ry + tri_list[index].z3 * sin_ry;
		tri_list[index].z3 = tmp_z;
		tri_list[index].x3 = tmp_x;

		//Z axis rotation, by 360-degree angle rz//
		tmp_x = tri_list[index].x3 * cos_rz - tri_list[index].y3 * sin_rz;
		tmp_y = tri_list[index].y3 * cos_rz + tri_list[index].x3 * sin_rz;
		tri_list[index].x3 = tmp_x;                                    
		tri_list[index].y3 = tmp_y;

		//Translate point origin back//
		tri_list[index].x3 += cx;
		tri_list[index].y3 += cy;
		tri_list[index].z3 += cz;		
	}
}


void MapGen::DistortMap(void)
{
	long index;
	for(index=0; index<num_triangles; index++)
	{
		float rand_x = (float)(rand()%2 ? -rand()%3 : rand()%3) / 2;
		float rand_y = (float)(rand()%2 ? -rand()%3 : rand()%3) / 2;
		float rand_z = (float)(rand()%2 ? -rand()%3 : rand()%3) / 2;

		tri_list[index].x1 = tri_list[index].x1 + rand_x;
		tri_list[index].y1 = tri_list[index].y1 + rand_y;
		tri_list[index].z1 = tri_list[index].z1 + rand_z;

		tri_list[index].x2 = tri_list[index].x2 + rand_x;
		tri_list[index].y2 = tri_list[index].y2 + rand_y;
		tri_list[index].z2 = tri_list[index].z2 + rand_z;

		tri_list[index].x3 = tri_list[index].x3 + rand_x;
		tri_list[index].y3 = tri_list[index].y3 + rand_y;
		tri_list[index].z3 = tri_list[index].z3 + rand_z;

		tri_list[index].color += (ULONG)rand_x;
	}
}


void MapGen::ScaleMap(float mag)
{
	long index;
	for(index=0; index<num_triangles; index++)
	{
		tri_list[index].x1 *= mag;
		tri_list[index].y1 *= mag;
		tri_list[index].z1 *= mag;

		tri_list[index].x2 *= mag;
		tri_list[index].y2 *= mag;
		tri_list[index].z2 *= mag;

		tri_list[index].x3 *= mag;
		tri_list[index].y3 *= mag;
		tri_list[index].z3 *= mag;
	}

	dimension = (long)(mag * (float)dimension);
}


void MapGen::DiamondStep(int x, int y, int halfstep, int randomness)
{
	float value;
	value = (GetValue(x - halfstep, y - halfstep) +
			 GetValue(x + halfstep, y - halfstep) +
			 GetValue(x - halfstep, y + halfstep) +
			 GetValue(x + halfstep, y + halfstep)) / 4; 
	
	if(rand() % 2)
	{
		value += (rand() % randomness);
	}
	else 
	{
		value -= (rand() % randomness);
	}

	SetValue(x, y, value); 
}


void MapGen::SquareStep(int x, int y, int halfstep, int randomness)
{
	float value;
	value = (GetValue(x - halfstep,y) +
			 GetValue(x,y - halfstep) +
			 GetValue(x,y + halfstep) +
			 GetValue(x + halfstep,y)) / 4;
	
	if(rand() % 2) 
	{
		value += (rand() % randomness);
	}
	else 
	{
		value -= (rand() % randomness);
	}

	SetValue(x, y, value);
}


void MapGen::SetValue(int x, int y, float value)
{
	if(x >= dimension) x -= dimension;
	else
	if(x < 0) x += dimension;
	
	if(y >= dimension) y -= dimension;
	else
	if(y < 0) y += dimension;

	height_map[x + (y * dimension)] = value;
}


float MapGen::GetValue(int x, int y)
{
	if(x >= dimension) x -= dimension;
	else
	if(x < 0) x += dimension;
	
	if(y >= dimension) y -= dimension;
	else
	if(y < 0) y += dimension;

	return height_map[x + (y * dimension)];
}


MapGen::~MapGen()
{
	iterations = 0;
	dimension = 0;
	num_triangles = 0;
	
	if(height_map != NULL) 
	{
		delete[] height_map;
		height_map = NULL;
	}

	if(tri_list != NULL)
	{
		delete[] tri_list;
		tri_list = NULL;
	}
}



#endif
